(ns halloween.spider.controller
  "Controller takes..."
  (:use [quil.core :only [dist key-as-keyword raw-key mouse-x mouse-y pmouse-x pmouse-y no-cursor cursor]]) ; ? necessary
  (:require [halloween.spider.renderer :as renderer]) ; switch to use only
  (:use [halloween.spider.renderer 
         :only [pan
                reset-pan
                scaler ; ? include scale-modifier and change scaler function to be more simple
                unhighlight
                highlight]])
  (:use [halloween.spider.simulator 
         :only [delete-procedure
                load-procedure
                save-procedure
                global-coords
                global-coords-next-location
                set-distance
                set-turn
                set-bend
                turn
                add-procedure
                toggle-silk
                bend-left
                bend-right
                reset-saved-info
                reset-preview
                concat-procedure
                add-bend-distance-turn-move-procedures
                move
                undo
                update-distance
                commit-preview-to-web]])
  (:use halloween.spider.api))

(def turn-angle 1) 
(def turn-angles [1 30 60 90])
(def mouse-enabled false)
(def mouse-pressed-once true)
(def pressed-mouse {:x 0 :y 0})
(def space false)
(def alt false)

; Window Event Handlers
;

(defn focus-gained
  []
  (def mouse-enabled true)
  )

(defn focus-lost
  []
  (def mouse-enabled false)
  (def mouse-pressed-once false)
  (unhighlight)
  )


; Interfaces to the Simulator and Renderer
;
  
(defn toggle-turn-angle
  [current-turn-angle]
  (let [x current-turn-angle]
    (if (= x 1) 30 (if (= x 30) 60 (if (= x 60) 90 (if (= x 90) 1))))
    )
  )

(defn set-scaler [n]
  ;(scaler (+ (scaler) (/ (* n (scaler)) 100)))
  (scaler (+ (scaler) n))
  (when (<= (scaler) 1)
    (scaler 1)
    )
  (when (>= (scaler) 100)
    (scaler 100)
    )
  )

(defn reset-view []
  (reset-pan)
  (scaler 30)
  )

(defn eval-input
  [input-to-eval add-to-procedure?]
  (try
    (remove-ns 'halloween.spider.api)
    (use 'halloween.spider.api :reload)
    (reset-saved-info)
    (reset-preview)
    (let [
          result (binding [*ns* (the-ns 'halloween.spider.api)]
                   (binding [*read-eval* false] (load-string input-to-eval)))]
      
    (if add-to-procedure? (do
                            (concat-procedure (vec (binding [*read-eval* false] (read-string (str "(" input-to-eval ")")))))
                            (commit-preview-to-web)
                            ))
    (str result))
    (catch Exception e (.getLocalizedMessage e)))
  )


; Mouse Event Handlers
;
  
(defn mouse-moved
  "Updates mouse position in relationship to the the renderer's window."
  []
  (when mouse-enabled
    (when space
      (pan {:x (- (mouse-x) (pmouse-x)) :y (- (mouse-y) (pmouse-y))})
      )
    (let [old-loc (global-coords)
          distance (dist (old-loc :x) (old-loc :y) (pmouse-x) (pmouse-y))
          angle (Math/toDegrees (Math/atan2 (- (old-loc :y)  (mouse-y)) (- (mouse-x) (old-loc :x))))
          divided-a (Math/round (/ angle turn-angle))
          rounded-distance (Math/round ( / distance (scaler)))]
      {
       :distance (if-not (= 0 rounded-distance) (set-distance rounded-distance) (set-distance 1))
       :orientation (set-turn (* divided-a turn-angle))
       }   
      ))
  )

(defn mouse-pressed 
  []
  (when mouse-pressed-once
    (when mouse-enabled
      (add-bend-distance-turn-move-procedures)
      (move)
      (commit-preview-to-web)
      (set-distance 1)
      (def pressed-mouse {:x (mouse-x) :y (mouse-y)})
      )
    )
  (when-not mouse-pressed-once
    (highlight)
    (def mouse-pressed-once true)
    )
  (no-cursor)
  )

(defn mouse-released 
  []
    (when-not mouse-pressed-once
    (def mouse-pressed-once true)
    (highlight)
    )
    (cursor)
  )

(defn mouse-dragged
  []
  (when (> (dist (mouse-x) (mouse-y) (pressed-mouse :x) (pressed-mouse :y)) (scaler))
    (mouse-moved)
    (mouse-pressed)
    )

  ;(let [location (global-coords-next-location)
        ;distance (dist (location :x) (location :y) (mouse-x) (mouse-y))
        ;old-loc (global-coords)
        ;p-distance (dist (location :x) (location :y) (pmouse-x) (pmouse-y))
        ;angle (Math/atan2 (- (old-loc :y) (mouse-y) ) (- (mouse-x) (old-loc :x)))
        ;difference (- (mouse-x) (pmouse-x))
        ;rotated-mousex (* (- (mouse-x) (location :x)) (Math/cos angle))
   ;     ]
    ;(set-bend (Math/round (- (mouse-x) (location :x))))
    ;)
  )


; Keyboard Event Handlers
;
  
(defn key-pressed []
  (case (key-as-keyword)
    :right   (do (turn (* turn-angle -1)) (add-procedure (list 'turn-right turn-angle)))
    :left    (do (turn turn-angle) (add-procedure (list 'turn-left turn-angle)))
    :up      (do (update-distance 1))
    :down    (do (update-distance -1))
    :command (do (def turn-angle (toggle-turn-angle turn-angle)))
    :shift   (do (toggle-silk))
    :z       (do (binding [*ns* (the-ns 'halloween.spider.simulator)] (undo))) ; ? is the binding necessary
    :a       (do (bend-left 5) (add-procedure (list 'bend-left 5)))
    :s       (do (bend-right 5) (add-procedure (list 'bend-right 5)))
    :x       (delete-procedure)
    :alt     (def alt true)
    nil)
  (case (int (raw-key))
    32       (def space true)
    nil)
  )

(defn key-released []
  (case (key-as-keyword) 
    :alt     (def alt false)
    nil)
  (case (int (raw-key)) 
    32       (def space false)
    nil))
