(ns halloween.spider.renderer
  (:use quil.core))

(def pan-modifier {:x 0 :y 0})
(def scale-modifier 30)
(def highlighted true)

(defn get-pan-mod [] pan-modifier)
(defn get-scaler [] scale-modifier)

(def show-circles true)


(defn setup 
  []
  (frame-rate 60)
  (smooth)
  (no-fill)
  (stroke 255)
  (stroke-weight 1.5)
  (cursor :arrow)
  )

(defn render-pdf [spider]
  
  (background 255)
  
  (translate (+ (/ (width) 2) (pan-modifier :x))
             (+ (/ (height) 2) (pan-modifier :y)))
  
  ;draw web
  
  (let [x   ((spider :location) :x)
        y   ((spider :location) :y)
        distance (spider :distance)
        orientation (spider :orientation)
        web (spider :web)
        silk (if (> (count web) 0) (:silk (meta (peek web))))]
        
    (doseq [strand web]
      (if (= (:silk (meta strand)) :on)
        (stroke 0 255)
        (stroke 0 50)
        )
      
      (push-matrix)
      
      ;(scale scale-modifier)
      
      (def t1 (random -1.0 1.0)) ; shake shake shake
      (def t2 (random -1.0 1.0))
      
      (translate t1 t2)
      
      (begin-shape)
      (doseq [location strand]
        (when (= (count strand) 1)
          (vertex (* scale-modifier (location :x))  (* scale-modifier (location :y)))
          )
        (when-not (location :curve)
          (vertex (* scale-modifier (location :x)) (* scale-modifier (location :y)))
          )
        (when (location :curve)
          (.bezierVertex 
            (quil.applet/current-applet) ; calling .bezierVertex due to bezier-vertex bug in quil
            (* scale-modifier (:x1 (:curve location)))
            (* scale-modifier (:y1 (:curve location)))
            (* scale-modifier (:x2 (:curve location)))
            (* scale-modifier (:y2 (:curve location)))
            (* scale-modifier (location :x))
            (* scale-modifier (location :y))
            )
          )
        )
      (end-shape)
      (pop-matrix))
  
    ;(if (= silk :on)
     ; (stroke 100 150 100)
      ;(stroke 100 150 100 100)
      ;)
    
    (exit)))

(defn render [spider]
  
  (background 0)
  
  ; draw laser dot
  (stroke 255 0 255 120)
  (line (pmouse-x) (pmouse-y) (mouse-x) (mouse-y))

  
  (translate (+ (/ (width) 2) (pan-modifier :x))
             (+ (/ (height) 2) (pan-modifier :y)))

  
  ;draw crosshair  
  (if highlighted
    (stroke 100 150 0)
    (stroke 50 50 0)
    )
  (let [s 2]
    (line (- s) (- s) s s)
    (line (- s) s s (- s))
    )
  
  (stroke 255 30)
  
  (line 0 (- (- (height)) (pan-modifier :y)) 0 (- (height) (pan-modifier :y)))
  (line (- (- (width)) (pan-modifier :x)) 0 (- (width) (pan-modifier :x)) 0)

  ;draw web

  (let [x   ((spider :location) :x)
        y   ((spider :location) :y)
        distance (spider :distance)
        orientation (spider :orientation)
        web (spider :web)
        silk (if (> (count web) 0) (:silk (meta (peek web))))]
    
  (stroke 255 100)

  (doseq [strand web]
    (if (= (:silk (meta strand)) :on)
      (stroke 255 200)
      (stroke 255 0 0 50)
      )
    
    (push-matrix)
    
    ;(scale scale-modifier)
    
    (def t1 (random -1.0 1.0)) ; shake shake shake
    (def t2 (random -1.0 1.0))
    
    (translate t1 t2)
    
    (begin-shape)
    (doseq [location strand]
      (when (= (count strand) 1)
        (vertex (* scale-modifier (location :x))  (* scale-modifier (location :y)))
        )
      (when-not (location :curve)
        (vertex (* scale-modifier (location :x)) (* scale-modifier (location :y)))
        )
      (when (location :curve)
        (.bezierVertex 
          (quil.applet/current-applet) ; calling .bezierVertex due to bezier-vertex bug in quil
          (* scale-modifier (:x1 (:curve location)))
          (* scale-modifier (:y1 (:curve location)))
          (* scale-modifier (:x2 (:curve location)))
          (* scale-modifier (:y2 (:curve location)))
          (* scale-modifier (location :x))
          (* scale-modifier (location :y))
          )
        )
      )
    (end-shape)
    (pop-matrix)
    )
 
  (if (= silk :on)
    (stroke 100 150 100)
    (stroke 100 150 100 100)
    )
  
  (push-matrix)
   
   (when (:curve (spider :location))
     (let [
           b-x1 (:x1 (:curve (spider :location)))
           b-y1 (:y1 (:curve (spider :location)))
           b-x2 (:x2 (:curve (spider :location)))
           b-y2 (:y2 (:curve (spider :location)))
           ]
       (bezier 
         (* scale-modifier ((peek (peek web)) :x))
         (* scale-modifier ((peek (peek web)) :y))
         (* scale-modifier b-x1)
         (* scale-modifier b-y1)
         (* scale-modifier b-x2)
         (* scale-modifier b-y2)
         (* scale-modifier x)
         (* scale-modifier y))
       )
     )
   
   (when-not (:curve (spider :location))
     (line 
       (* scale-modifier ((peek (peek web)) :x) )
       (* scale-modifier ((peek (peek web)) :y) )
       (* scale-modifier x)
       (* scale-modifier y)) 
     )
  


   ; Circles
   
   (when show-circles
     (when (> (count (peek web)) 1)
       (stroke 255 0 0 40)
       (ellipse (* scale-modifier ((peek (pop (peek web))) :x) ) (* scale-modifier ((peek (pop (peek web))) :y) ) (* scale-modifier 2) (* scale-modifier 2))
       ;(ellipse (* scale-modifier ((peek (pop (peek web))) :x) ) (* scale-modifier ((peek (pop (peek web))) :y) ) (* scale-modifier 4) (* scale-modifier 4))
       )
     
     (stroke 255 0 255 40)
     (ellipse (* scale-modifier ((peek (peek web)) :x) ) (* scale-modifier ((peek (peek web)) :y) ) (* scale-modifier 2) (* scale-modifier 2))
     (ellipse (* scale-modifier ((peek (peek web)) :x) ) (* scale-modifier ((peek (peek web)) :y) ) (* scale-modifier 4) (* scale-modifier 4))
     
     (stroke 255 255 0 40)
     (ellipse (* scale-modifier x) (* scale-modifier y) (* scale-modifier 2) (* scale-modifier 2))
     (ellipse (* scale-modifier x) (* scale-modifier y) (* scale-modifier 4) (* scale-modifier 4))    
     )
   
   (pop-matrix)
   
 
   ; spider icon
   (translate 
    (* scale-modifier x) 
    (* scale-modifier y)
    )
   
   ;(translate 
   ;  (* scale-modifier ((peek (peek web)) :x))
   ;  (* scale-modifier ((peek (peek web)) :y)))
   
   (rotate (- orientation))
   
   (let [size 20
         quarter-size (* size 1/4)
         three-quarter-size (* size 3/4)]
     
     (stroke 255 0 0)
     
     ;pincers
     (line (+ (* size 3/2)) 0 (+ (* size 3/2) (/ size 5)) (+ (/ size  10)))
     (line (+ (* size 3/2)) 0 (+ (* size 3/2) (/ size 5)) (+ (/ size -10)))
     
     ;left legs
     (line (+ (/ size 2)) 0 (- quarter-size) (- three-quarter-size))
     (line (+ (/ size 2)) 0 (+ quarter-size) (- three-quarter-size))
     (line (+ (/ size 2)) 0 (+ three-quarter-size) (- three-quarter-size))
     (line (+ (/ size 2)) 0 (+ size quarter-size) (- three-quarter-size))
     
     ;right legs
     (line (+ (/ size 2)) 0 (- quarter-size) (+ three-quarter-size))
     (line (+ (/ size 2)) 0 (+ quarter-size) (+ three-quarter-size))
     (line (+ (/ size 2)) 0 (+ three-quarter-size) (+ three-quarter-size))
     (line (+ (/ size 2)) 0 (+ size quarter-size) (+ three-quarter-size))
     
     ;body
     (fill 80 0 0)
     (ellipse (+ (/ size 2)) 0 size size)
     (ellipse (+ (/ size 2) (* size 3/4) ) 0 (/ size 2) (/ size 2))
     (no-fill)
     
     
     ;cutters
     (when (= silk :off)
       (stroke 255 0 0)
       (line 0 0 (* -1 quarter-size) (+ (/ size  8)))
       (line 0 0 (* -1 quarter-size) (+ (/ size -8))))
     )
   ))

(defn get-width [] (width))
(defn get-height [] (height))

(defn highlight []
  (def highlighted true)
  )

(defn unhighlight []
  (def highlighted false)
  )

(defn pan [& amount]
  (if (nil? amount)
    pan-modifier 
    (def pan-modifier (merge-with + pan-modifier (first amount)))
    )
  )

(defn reset-pan []
  (def pan-modifier {:x 0 :y 0})
  )

(defn toggle-circles []
  (def show-circles (not show-circles))
  )

(defn scaler [& amount]
  (if (nil? amount) 
      scale-modifier 
      (def scale-modifier (first amount)))
  )

(defn export-procedure-as-pdf
  "Exports the drawing to a .pdf file."
  [file-path]
  
  
  ;(quil.applet/current-applet)
  ;(begin-record :pdf "test.pdf")
  ;(stroke 0 0 0)
  ;(line 10 100 10 100)
  ;(end-record)
  )