(ns halloween.spider.simulator
  (:use [clojure.pprint :only [pprint with-pprint-dispatch code-dispatch]]))

(def silk-state :off)
(def orientation 0)
(def scale 1)
(def distance 1)
(def bend-level 0)
(def strand (with-meta [{:x 0 :y 0}] {:silk :off}))
(def web [])
(def preview-web [])
(def saved-strand strand)
(def saved-preview-web [])
;(def preview-strand [])

(def saved-info {:orientation orientation
                 :silk silk-state
                 :distance distance
                 :bend-level bend-level
                 })

(def procedure [])

; Functions for Working w/ Strands and Webs
;

(defn empty-strand
  []
  (with-meta [{:x 0 :y 0}] {:silk :off})
  )

(defn add-strand-to-web
  [silk-state]
  (when (> (count strand) 1)
    (def preview-web (conj preview-web strand))
    )
  (def strand (with-meta [(dissoc (peek strand) :curve) ] {:silk silk-state}))
  )

(defn reset-strand
  []
  (when (> (count web) 0)
  (def strand [(peek (peek web))])
  )
  (when (= (count web) 0)
    (def strand (empty-strand))
    )
  )

(defn reset-preview
  []
  (def strand saved-strand)
  (def preview-web [])
  (def silk-state (saved-info :silk))
  (def distance (saved-info :distance))
  (def orientation (saved-info :orientation))
  (def bend-level (saved-info :bend-level))
  )


; Functions for Working w/ Procedures
;

(defn concat-procedure [p] (def procedure (into [] (concat procedure p))))
(defn add-procedure [p] (def procedure (conj procedure p)))

(defn reset-saved-info
  []
  (def saved-info {
                   :orientation 0
                   :silk :off
                   :distance 1
                   :bend-level 0
                   })
  )

(defn reset-web
  []
  (reset-preview)
  (def strand (empty-strand))
  (def web [strand])
  )

(defn delete-procedure
  []
  (reset-saved-info)
  (reset-preview)
  (def preview-web [])
  (reset-web)
  (def saved-strand (empty-strand))
  (def saved-preview-web [])
  (def procedure [])
  (def preview-web [])
  (def web [])
  )

(defn commit-preview-to-web
  []
  
  (let 
    [
     temp-strand (with-meta (into [] (concat (last preview-web) strand)) (meta strand))
     ]
    
    (def saved-preview-web preview-web)
    (def saved-strand strand) ;maybe this needs fixing - the saved info?
    (def saved-info {
                     :orientation orientation
                     :silk silk-state
                     :distance distance
                     :bend-level bend-level
                     })
    
    (when (>= (count preview-web) 1)
      (def web (into [] (concat web preview-web)))
      (def preview-web []))))

(defn execute-procedure
  "Executes a set of spider commands."
  [p]
  (doseq [x p] 
    (binding [*ns* (the-ns 'halloween.spider.simulator)] (eval x))
    )
  (commit-preview-to-web)
  )


(defn undo
  []
  (when (seq procedure)
    (reset-saved-info)
    (def procedure (pop procedure))
    (reset-web)
    (execute-procedure procedure)
    )
  )

(defn save-procedure 
  "Saves drawing results to .clj file."
  [file-path]
  (spit file-path (with-out-str (with-pprint-dispatch code-dispatch (pprint procedure))))
  )

(defn load-procedure
  "Loads .clj file as a list."
  [file-path]
  (try
    (println file-path)
    (reset-preview)
    (reset-saved-info)
    (reset-web) 
    (def procedure (binding [*read-eval* false] (read-string (slurp file-path))))
    (execute-procedure procedure)
    (catch Exception e (.getLocalizedMessage e)))
  )

(defn add-bend-distance-turn-move-procedures ; eventually use a let and add all procedures at once
  []  
  (when (:orientation saved-info)
    (when-not (= orientation (:orientation saved-info))
      (let [
            new-orientation (- orientation (:orientation saved-info))
            ]
        (if (< new-orientation 0)
        (add-procedure (list 'turn-right (- new-orientation)))
        (add-procedure (list 'turn-left new-orientation)))
        )
      )
    )
  
  (when (:bend-level saved-info)
    (when-not (= bend-level (:bend-level saved-info))
      (let [
            new-bend-level (- bend-level (:bend-level saved-info))
            ]
        (add-procedure (list 'bend new-bend-level))
        )
      )
    )
  
  (when (:distance saved-info)
    (when-not (= distance (:distance saved-info))
      (let [
            new-distance (- distance (:distance saved-info))
            ]
        (add-procedure (list 'forward distance))
        )
      )
    )
  
  ;(when (<= (count procedure) 1)
   ; (add-procedure (list 'turn orientation))
    ;(if-not (= 0 bend-level) (add-procedure (list 'bend bend-level)))
    ;(add-procedure (list 'forward distance))
    ;)
  
  (when (= distance (:distance saved-info))
    (add-procedure (list 'forward))
    ))


; Change spider's state
;

(defn update-distance ; possibly make better?
  [amount]
  (def distance (+ distance amount))
  (if (<= distance 1)
   (def distance 1))
  )

(defn set-distance
  [amount]
  @(def distance amount)
  )

(defn set-turn
  [amount]
  @(def orientation amount)
  )

(defn set-bend
  [amount]
  (def bend-level amount)
  )

(defn bend
  [amount]
  (def bend-level (+ bend-level amount))
  )

(defn bend-right
  [amount]
  (def bend-level (- bend-level amount))
)

(defn bend-left
  [amount]
  (def bend-level (+ bend-level amount))
  )

(defn curve-line
  [b coords]
   
  (let
    [
     x1 (+ (:x1 coords) (* (/ distance 2) (Math/cos (Math/toRadians orientation))))
     y1 (- (:y1 coords) (* (/ distance 2) (Math/sin (Math/toRadians orientation))))
     x2 (+ x1 (* b (Math/cos (Math/toRadians (+ orientation 90)))))
     y2 (- y1 (* b (Math/sin (Math/toRadians (+ orientation 90)))))
     bx1 (/ (+ (:x1 coords) (* 2 x2)) 3)
     by1 (/ (+ (:y1 coords) (* 2 y2)) 3)
     bx2 (/ (+ (:x2 coords) (* 2 x2)) 3)
     by2 (/ (+ (:y2 coords) (* 2 y2)) 3)]
  
    {:x1 bx1 :y1 by1 :x2 bx2 :y2 by2}))

(defn get-new-location
  [location new-distance]
  (let
    [x (+ (location :x) (* new-distance (Math/cos (Math/toRadians orientation))))
     y (- (location :y) (* new-distance (Math/sin (Math/toRadians orientation))))
     new-location {
                   :x x
                   :y y
                   :distance new-distance
                   :orientation orientation
                   :bend-level bend-level
                   }]
    
     (if-not (= bend-level 0)
       (assoc new-location :curve
              (curve-line
                bend-level
                {
                 :x1 (location :x) 
                 :y1 (location :y)
                 :x2 x
                 :y2 y
                 }
                ))
       new-location
       )
    )
  )

(defn turn ; default turns right 90 degrees 
  [& angle]
  (if (nil? angle) (def new-angle -90) (def new-angle (first angle)))  
  (let [new-orientation (+ orientation new-angle)]
    (if (<= new-orientation 0)
      (def orientation (+ new-orientation 360))
          (if (>= new-orientation 360)
      (def orientation (- new-orientation 360))
      (def orientation new-orientation)
      )
      )
    )
  )

(defn global-coords-next-location []
  
  (let [
        pan (halloween.spider.renderer/get-pan-mod)
        scale (halloween.spider.renderer/get-scaler)
        location (get-new-location (peek strand) distance)
        translated-x (+ (location :x) (/ (halloween.spider.renderer/get-width) 2) (pan :x))
        translated-y (+ (location :y) (/ (halloween.spider.renderer/get-height) 2) (pan :y))
        ]
    {:x translated-x :y translated-y}
    ))

(defn global-coords []
  
  (let [
        pan (halloween.spider.renderer/get-pan-mod)
        scale (halloween.spider.renderer/get-scaler)
        location (peek strand)
        translated-x (+ (* scale (location :x)) (/ (halloween.spider.renderer/get-width) 2) (pan :x))
        translated-y (+ (* scale (location :y)) (/ (halloween.spider.renderer/get-height) 2) (pan :y))
        t-x (/ translated-x scale)
        t-y (/ translated-y scale)
        ]
    {:x translated-x :y translated-y}
    ))

(defn silk ; combine toggle-silk and silk
  [state]
  (if (> (count strand) 0) 
    (if-not (= (:silk (meta strand)) state)
      (add-strand-to-web state)
      ))
  )

(defn switch-silk
  []
  (if (= silk-state :on)
    (def silk-state :off)
    (def silk-state :on)
    )  
  (if (> (count strand) 0) ; maybe repair this?
    (if-not (= (:silk (meta strand)) silk-state)
      (add-strand-to-web silk-state)
      ))
  )

(defn move
  [& amount]
  (if (nil? amount) (def new-distance distance) (def new-distance (first amount)))
  (if-not (= 0 new-distance)
    (do
      (def distance new-distance)
      (let [new-strand (get-new-location (peek strand) new-distance)]
        (if-not (= strand new-strand) (def strand (conj strand new-strand))))
      )
    "Cannot travel a distance of 0"
    )
  )

(defn toggle-silk ; maybe repair this
  []
  (switch-silk)
  (let [length (count procedure)]
    (when (>= length 2)
      (let [last-two-procedures (take-last 2 procedure)] 
        (if-not (= last-two-procedures [(list 'switch-silk) (list 'switch-silk)])
          (add-procedure (list 'switch-silk))
          (def procedure (pop procedure)))))
    (when (<= length 1)
      (add-procedure (list 'switch-silk))))
  )


; API Definitions
;

(defn m [n] (move n))

(defn forward [& n] 
  (if n 
    (move (first n))
    (move)))
(def f forward)

(defn backward [& n] 
  (if n 
    (move (- (first n)))
    (move (- distance))))
(def back backward)
(def b backward)

(defn silk [] (switch-silk))
(defn s [] (silk))

(defn t [& n] 
  (if n 
    (turn (first n))
    (turn 90))
  )

(def turn-left t)

(defn turn-right [& n] 
  (if n 
    (turn (- (first n)))
    (turn -90))
  )

(def tr turn-right)
(def tl turn-left)


(defn curve [& n] 
  (if n 
    (bend (first n))
    (bend 10))
  )

(def c curve)

(def curve-left c)
(def cl curve-left)

(defn curve-right [& n] 
  (if n 
    (bend (- (first n)))
    (bend -10))
  )
(def cr curve-right)

(defn retract [& n]
  (switch-silk)
  (if n
    (backward n)
    (backward distance))
  (switch-silk))

(def r retract)

(defn simulate
  "Simulates one step and returns display coordinates."
  []
  {;:preview-web preview-web
   :location (get-new-location (peek strand) distance) ;combine with global-coords
   :orientation (Math/toRadians orientation)
   :distance distance
   :web (into [] (concat web (conj preview-web strand)))
   }
  )