(ns halloween.spider.ui
  "User Interface Elements"
  (:use [quil.core :only [defsketch exit]])
  (:use [clojure.java.io :only [input-stream]])
  (:use [seesaw.core]
        [seesaw.keymap]
        [seesaw.chooser]
        [seesaw.widget-options :only [widget-option-provider]]
        [seesaw.options :only [bean-option option-map apply-options]])
  (:import [org.fife.ui.rtextarea RTextScrollPane Gutter])
  (:import [org.fife.ui.rsyntaxtextarea RSyntaxTextArea Theme SyntaxConstants])
  (:use [halloween.spider.renderer
         :only [setup
                render
                toggle-circles
                render-pdf]])
  (:use [halloween.spider.simulator 
         :only [simulate
                delete-procedure
                load-procedure
                save-procedure
                reset-preview]])
  (:use [halloween.spider.controller
         :only [eval-input
                key-pressed
                key-released
                mouse-pressed
                mouse-released
                mouse-dragged
                mouse-moved
                focus-lost
                focus-gained
                set-scaler
                reset-view
                focus-gained]]))

(native!)

(def current-file "drawing.clj")

; Integrate RSyntaxTextArea into Seesaw & Define Brain Input
;

(def code-area-options 
  (merge 
    text-area-options 
    (option-map 
      (bean-option :anti-aliasing-enabled RSyntaxTextArea))))
(widget-option-provider RSyntaxTextArea code-area-options) 

(defn code-area 
  [& args] 
  (case (count args) 
    0 (code-area :text "") 
    1 (code-area :text (first args)) 
    (apply-options (construct RSyntaxTextArea) args))) 

(defn- make-source-editor [] 
  (doto (code-area 
          :id :code-area 
          :anti-aliasing-enabled true 
          :text ""
          ) 
    (.setSyntaxEditingStyle SyntaxConstants/SYNTAX_STYLE_CLOJURE)
    (.setAutoIndentEnabled true)
    (.setTabSize 2)
    ))

(def input-field (make-source-editor))


; Brain Output (console)
;
  
(def output-field (text :text "?" :multi-line? true))
(defn print-status [status] (text! output-field (str status)))
(def remove-input-listener
  (listen input-field :document (fn [e] (text! output-field (eval-input (text input-field) false))))
  )


; Export
;

(defn export-pdf
  "Exports drawing as a .pdf file."
  [file-name]
  (def fname file-name)
  (defsketch pdf-export
    :title "Exporting PDF"
    :renderer :pdf
    :output-file (str fname)
    :setup setup
    :draw (fn [] 
            (render-pdf (simulate)))
    :size [800 600])
  (print-status (str fname))
  )


; Application Menu
;

(def brain-menu-bar
    
  (menubar :items 
           [(menu :text "File" :items [(action
                                         :name "New"
                                         :key  "menu X"
                                         :tip  "Create a new file."
                                         :handler (fn [e] 
                                                    (delete-procedure)))
                                       (action
                                         :name "Open"
                                         :key  "menu O"
                                         :tip  "Open a file."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :open 
                                                      :filters [["Clojure (.clj)" ["clj"]]] 
                                                      :success-fn (fn [fc file] 
                                                                    (def current-file (.getAbsolutePath file)) 
                                                                    (load-procedure current-file)))))
                                       (action
                                         :name "Save"
                                         :key  "menu S" 
                                         :tip  "Overwrites the currently loaded .clj file."
                                         :handler (fn [e] 
                                                    (save-procedure current-file)
                                                    (print-status (str "Saved " current-file))))
                                       (action
                                         :name "Save as..."
                                         :tip  "Saves .clj file as specified name."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :save 
                                                      :filters [["Clojure (.clj)" ["clj"]]] 
                                                      :success-fn (fn [fc file]
                                                                    (def current-file (str (.getAbsolutePath file) ".clj"))
                                                                    (save-procedure current-file)))))
                                       (action
                                         :name "Export PDF"
                                         :key  "menu E" 
                                         :tip  "Exports a .pdf file of the drawing."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :save 
                                                      :filters [["Portable Document Format (.pdf)" ["pdf"]]] 
                                                      :success-fn (fn [fc file]
                                                                    (def current-file (str (.getAbsolutePath file) ".pdf"))
                                                                    (export-pdf current-file)))))])
            (menu :text "View" :items [(action
                                         :name "Reset"
                                         :key "menu R"
                                         :tip  "Resets the viewport."
                                         :handler (fn [e]
                                                    (reset-view)))
                                       (action
                                         :name "Toggle Protactors"
                                         :key "menu C"
                                         :tip  "Toggles Protractors"
                                         :handler (fn [e] 
                                                    (toggle-circles)))])]))

(def habitat-menu-bar
  
  (menubar :items 
           [(menu :text "File" :items [(action
                                         :name "New"
                                         :key  "menu X"
                                         :tip  "Create a new file."
                                         :handler (fn [e] 
                                                    (delete-procedure)))
                                       (action
                                         :name "Open"
                                         :key  "menu O"
                                         :tip  "Open a file."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :open 
                                                      :filters [["Clojure (.clj)" ["clj"]]] 
                                                      :success-fn (fn [fc file] 
                                                                    (def current-file (.getAbsolutePath file)) 
                                                                    (load-procedure current-file)))))
                                       (action
                                         :name "Save"
                                         :key  "menu S" 
                                         :tip  "Overwrites the currently loaded .clj file."
                                         :handler (fn [e] 
                                                    (save-procedure current-file)
                                                    (print-status (str "Saved " current-file))))
                                       (action
                                         :name "Save as..."
                                         :tip  "Saves .clj file as specified name."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :save 
                                                      :filters [["Clojure (.clj)" ["clj"]]] 
                                                      :success-fn (fn [fc file]
                                                                    (def current-file (str (.getAbsolutePath file) ".clj"))
                                                                    (save-procedure current-file)))))
                                       (action
                                         :name "Export PDF"
                                         :key  "menu E" 
                                         :tip  "Exports a .pdf file of the drawing."
                                         :handler (fn [e] 
                                                    (choose-file 
                                                      :type :save 
                                                      :filters [["Portable Document Format (.pdf)" ["pdf"]]] 
                                                      :success-fn (fn [fc file]
                                                                    (def current-file (str (.getAbsolutePath file) ".pdf"))
                                                                    (export-pdf current-file)))))])
            (menu :text "View" :items [(action
                                         :name "Reset"
                                         :key "menu R"
                                         :tip  "Resets the viewport."
                                         :handler (fn [e]
                                                    (reset-view)))
                                       (action
                                         :name "Toggle Protactors"
                                         :key "menu C"
                                         :tip  "Toggles Protractors"
                                         :handler (fn [e] 
                                                    (toggle-circles)))])]))


; Opening and Closing UI Elements
;

(defn habitat
  []
  (defsketch spider-window
    :title "Habitat"
    :setup setup
    :draw (fn [] (render (simulate)))
    :target :none
    :key-pressed  key-pressed
    :key-released key-released
    :mouse-pressed mouse-pressed
    :mouse-released mouse-released
    :mouse-dragged mouse-dragged
    :mouse-moved (fn [] (print-status (mouse-moved)))
    :focus-lost focus-lost
    :renderer :java2d
    :size [800 600])
  (listen spider-window :mouse-wheel-moved (fn [e] (set-scaler (.getWheelRotation e))))
  (listen spider-window :focus-gained (fn [e] (focus-gained)))
  
  (def habitat-frame (frame :title "Habitat"
                            :width 800
                            :height 600
                            :minimum-size [640 :by 480]
                            :resizable? true
                            :content spider-window
                            :menubar habitat-menu-bar))
   
  (.setLocation habitat-frame 0 0)
  
  (-> habitat-frame show!)
  (request-focus! spider-window)
  )

(defn brain
  []
  
  (.apply (Theme/load (input-stream "theme/dark.xml")) input-field)
  
  (listen input-field :focus-gained (fn [e]
                                      (text! output-field (eval-input (text input-field) false))))
  
    (listen input-field :focus-lost (fn [e] (reset-preview)))
  
  (map-key input-field "ctrl ENTER" (fn [e]
                                      (remove-input-listener)
                                      (text! output-field (eval-input (text input-field) true))
                                      (text! input-field "");
                                      (def remove-input-listener
                                        (listen input-field :document (fn [e] (text! output-field (eval-input (text input-field) false))))
                                        )))
  
  (config! output-field 
           :font "MONOSPACED-PLAIN-14" 
           :wrap-lines? true 
           :editable? false 
           :border 4 
           :background "#333" 
           :foreground "#ddd" 
           :selection-color "#ccc")
  
  (let [content (top-bottom-split (scrollable input-field :border 4 :background "#545454") (scrollable output-field :border 4 :background "#333") :border 0 :background "#aaa")]
    (.setDividerSize content 2)
    (.setContinuousLayout content false)
    (.setResizeWeight content 1)
    (.setDividerLocation content 100)
    
    (def brain-frame 
      (frame :title "Brain"
               :width 600
               :height 160
               :minimum-size [800 :by 160]
               :resizable? true
               :content content
               :menubar brain-menu-bar
               ))
    
    (-> brain-frame show!))
  
  (.setLocation brain-frame 0 622)
  
  ) ; ? fix

; Initializing The UI
;

(defn init []
  (habitat)
  (brain)
  )

