(ns halloween.spider.simulator
  (:require [clojure.math.numeric-tower :as math])
  (:use clojure.pprint))

(def pen-state :off)
(def orientation 0)
(def scale 1)
(def slack 10)
(def bend-level 0)
(def strand (with-meta [{:x 0 :y 0}] {:pen :off}))
(def web [])
(def preview-web [])
(def saved-strand (with-meta [{:x 0 :y 0}] {:pen :off}))
(def saved-preview-web [])
;(def preview-strand [])

          (def saved-info {
                           :orientation orientation
                           :pen pen-state
                           :slack slack
                           :bend-level bend-level
                           })


(def procedure [])

(defn update-slack ; possibly make better?
  [amount]
  (def slack (+ slack amount))
  (if (<= slack 1)
   (def slack 1))
  )

(defn set-slack
  [amount]
  (def slack amount)
  )

(defn set-turn
  [amount]
  (def orientation amount)
  )

(defn set-bend
  [amount]
  (def bend-level amount)
  )

(defn bend
  [amount]
  (def bend-level (+ bend-level amount))
  )

(defn bend-right
  [amount]
  (def bend-level (- bend-level amount))
)

(defn bend-left
  [amount]
  (def bend-level (+ bend-level amount))
  )

(defn curve-line
  [b coords]
   
  (let
    [
     x1 (+ (:x1 coords) (* (/ slack 2) (Math/cos (Math/toRadians orientation))))
     y1 (- (:y1 coords) (* (/ slack 2) (Math/sin (Math/toRadians orientation))))
     x2 (+ x1 (* b (Math/cos (Math/toRadians (+ orientation 90)))))
     y2 (- y1 (* b (Math/sin (Math/toRadians (+ orientation 90)))))
     bx1 (/ (+ (:x1 coords) (* 2 x2)) 3)
     by1 (/ (+ (:y1 coords) (* 2 y2)) 3)
     bx2 (/ (+ (:x2 coords) (* 2 x2)) 3)
     by2 (/ (+ (:y2 coords) (* 2 y2)) 3)]
  
    {:x1 bx1 :y1 by1 :x2 bx2 :y2 by2}))

(defn get-new-location
  [location new-slack]
  (let
    [x (+ (location :x) (* new-slack (Math/cos (Math/toRadians orientation))))
     y (- (location :y) (* new-slack (Math/sin (Math/toRadians orientation))))
     new-location {
                   :x x
                   :y y
                   :slack new-slack
                   :orientation orientation
                   :bend-level bend-level
                   }]
    
     (if-not (= bend-level 0)
       (assoc new-location :curve
              (curve-line
                bend-level
                {
                 :x1 (location :x) 
                 :y1 (location :y)
                 :x2 x
                 :y2 y
                 }
                ))
       new-location
       )
    )
  )

(defn turn ; default turns right 90 degrees 
  [& angle]
  (if (nil? angle) (def new-angle -90) (def new-angle (first angle)))  
  (let [new-orientation (+ orientation new-angle)]
    (if (<= new-orientation 0)
      (def orientation (+ new-orientation 360))
          (if (>= new-orientation 360)
      (def orientation (- new-orientation 360))
      (def orientation new-orientation)
      )
      )
    )
  )

(defn global-coords-next-location []
  
  (let [
        pan (halloween.spider.renderer/get-pan-mod)
        scale (halloween.spider.renderer/get-scaler)
        location (get-new-location (peek strand) slack)
        translated-x (+ (location :x) (/ (halloween.spider.renderer/get-width) 2) (pan :x))
        translated-y (+ (location :y) (/ (halloween.spider.renderer/get-height) 2) (pan :y))
        ]
    {:x translated-x :y translated-y}
    ))

(defn global-coords []
  
  (let [
        pan (halloween.spider.renderer/get-pan-mod)
        scale (halloween.spider.renderer/get-scaler)
        location (peek strand)
        translated-x (+ (* scale (location :x)) (/ (halloween.spider.renderer/get-width) 2) (pan :x))
        translated-y (+ (* scale (location :y)) (/ (halloween.spider.renderer/get-height) 2) (pan :y))
        t-x (/ translated-x scale)
        t-y (/ translated-y scale)
        ]
    {:x translated-x :y translated-y}
    ))

(defn get-procedure
  "Returns the procedure."
  []
  procedure)

(defn add-procedure [p] (def procedure (conj procedure p)))

(defn concat-procedure [p]
  (def procedure (into [] (concat procedure p)))
  )


(defn add-strand-to-web
  [pen-state]
  (when (> (count strand) 1)
  (def preview-web (conj preview-web strand))
  )
  (def strand (with-meta [
                          (dissoc (peek strand) :curve) ] {:pen pen-state}))
  )

(defn pen ; combine toggle-pen and pen
  [state]
  (if (> (count strand) 0) 
    (if-not (= (:pen (meta strand)) state)
      (add-strand-to-web state)
    ))
)

(defn switch-pen
  []
  (if (= pen-state :on)
  (def pen-state :off)
  (def pen-state :on)
  )
  
  (if (> (count strand) 0) ; maybe repair this?
   (if-not (= (:pen (meta strand)) pen-state)
      (add-strand-to-web pen-state)
    ))
    )

(defn toggle-pen ; maybe repair this
  []
  (switch-pen)
  (let [length (count procedure)]
    (when (>= length 2)
      (let [
            last-two-procedures (take-last 2 procedure)]
        
        (if-not (= last-two-procedures [(list 'switch-pen) (list 'switch-pen)])
          (add-procedure (list 'switch-pen))
          (def procedure (pop procedure))
          )
        ) 
      )
    (when (<= length 1)
      (add-procedure (list 'switch-pen))
      )
    )
  )

(defn move
  [& amount]
  (if (nil? amount) (def new-slack slack) (def new-slack (first amount)))
  ;(def strand (conj strand (get-new-location {:x (:x (peek strand)) :y (:y (peek strand))} new-slack)))
  (def slack new-slack)
  (def strand (conj strand (get-new-location (peek strand) new-slack)))
  )

(defn reset-preview
  []
  (def slack 10)
  (def orientation 0)
  (def bend-level 0)
  )

(defn reset-strand
  []
  (when (> (count web) 0)
  (def strand [(peek (peek web))])
  )
  (when (= (count web) 0)
    (def strand (with-meta [{:x 0 :y 0}] {:pen :off}))
    )
  )

(defn reset-strand-draft
  []
  (def strand (with-meta [{:x 0 :y 0}] {:pen :off}))
  )

(defn reset-preview-parameters
  []
  ;(reset-strand)
  (def strand saved-strand)
  (def preview-web [])
  (def pen-state (saved-info :pen))
  (def slack (saved-info :slack))
  (def orientation (saved-info :orientation))
  (def bend-level (saved-info :bend-level))
  )

(defn reset-saved-info
  []
  (def saved-info {
                   :orientation 0
                   :pen :off
                   :slack 10
                   :bend-level 0
                   })
  )

(defn reset-procedure
  []
  (reset-preview-parameters)
  (def strand (with-meta [{:x 0 :y 0}] {:pen :off}))
  (def web [strand])
  )

(defn reset-preview-web-and-strand []
  (reset-preview-parameters)
  (def preview-web [])
  )

(defn delete-procedure
  []
  (reset-saved-info)
  (reset-preview-web-and-strand)
  (reset-procedure)
  (def saved-strand (with-meta [{:x 0 :y 0}] {:pen :off}))
  (def saved-preview-web [])
  (def procedure [])
  (def preview-web [])
  (def web [])
  )

  (defn commit-preview-to-web
    []

    (let 
      [
       temp-strand (with-meta (into [] (concat (last preview-web) strand)) (meta strand))
       ]
      
        (def saved-preview-web preview-web)
          (def saved-strand strand) ;maybe this needs fixing - the saved info?
          (def saved-info {
                           :orientation orientation
                           :pen pen-state
                           :slack slack
                           :bend-level bend-level
                           })
                 
        (when (>= (count preview-web) 1)
          (def web (into [] (concat web preview-web)))
          (def preview-web []))))
  


  (defn execute-procedure
   "Executes a set of spider commands."
   [p]
   (doseq [x p] 
     (binding [*ns* (the-ns 'halloween.spider.simulator)] (eval x))
     )
   (commit-preview-to-web)
   )
  
    (defn undo
    []
    (when (seq procedure)
      (reset-saved-info)
      (def procedure (pop procedure))
      (reset-procedure)
      (execute-procedure procedure)
      )
    )


  (defn save-procedure 
    "Saves drawing results to .clj file."
    [file-path]
    (spit file-path (with-out-str (with-pprint-dispatch code-dispatch (pprint procedure))))
    ;(spit "saved-drawing-test.clj" (with-out-str (clojure.pprint/pprint (simulator/get-procedure))))
    )
  
  (defn load-procedure
   "Loads .clj file as a list."
   [file-path]
   (try
     (println file-path)
     (reset-preview-parameters)
     (reset-saved-info)
     (reset-procedure) 
     (def procedure (binding [*read-eval* false] (read-string (slurp file-path))))
     (execute-procedure procedure)
     (catch Exception e (.getLocalizedMessage e)))
   )
  
  (defn open
    "Loads a .clj file as a working copy."
    [name]
    
    )
  
  (defn save
    "Writes out the working copy to a specified filename."
    [name]
    
    )
  
  (defn add-bend-slack-turn-move-procedures ; eventually use a let and add all procedures at once
  []
    
  (when (:orientation saved-info)
    (when-not (= orientation (:orientation saved-info))
      (let [
            new-orientation (- orientation (:orientation saved-info))
            ]
        (add-procedure (list 'turn new-orientation))
        )
      )
    )
  
    (when (:bend-level saved-info)
    (when-not (= bend-level (:bend-level saved-info))
      (let [
            new-bend-level (- bend-level (:bend-level saved-info))
            ]
        (add-procedure (list 'bend new-bend-level))
        )
      )
    )
  
  (when (:slack saved-info)
    (when-not (= slack (:slack saved-info))
      (let [
            new-slack (- slack (:slack saved-info))
            ]
        (add-procedure (list 'move slack))
        )
      )
    )
  
  (when (<= (count procedure) 1)
    (add-procedure (list 'turn orientation))
    (if-not (= 0 bend-level) (add-procedure (list 'bend bend-level)))
    (add-procedure (list 'move slack))
    )
  
  (when (= slack (:slack saved-info))
    (add-procedure (list 'move))
  ))
  
  (defn output-status []
    {:heading orientation
     :distance slack
     }
    )
  
  ; shorthand functions / exposable api
  (defn m [n] (move n))
  
  (defn forward [& n] 
    (if n 
      (move (first n))
      (move)))
  (def f forward)
  
  (defn backward [& n] 
    (if n 
      (move (- (first n)))
      (move (- slack))))
  (def back backward)
  (def b backward)
  
  (defn silk [] (switch-pen))
  (defn s [] (silk))
     
  (defn t [& n] 
    (if n 
      (turn (first n))
      (turn 90))
    )
  
  (def turn-left t)
  
  (defn turn-right [& n] 
    (if n 
      (turn (- (first n)))
      (turn -90))
    )
  
  (def tr turn-right)
  (def tl turn-left)

  
  (defn curve [& n] 
    (if n 
      (bend (first n))
      (bend 10))
    )
  
  (def c curve)
  
  (def curve-left c)
  (def cl curve-left)
  
    (defn curve-right [& n] 
    (if n 
      (bend (- (first n)))
      (bend -10))
    )
  (def cr curve-right)
  
  (defn retract [& n]
    (switch-pen)
    (if n
    (backward n)
    (backward slack))
    (switch-pen))
  
  (def r retract)
  
(defn simulate
  "Simulates one step and returns display coordinates."
  []
  {;:preview-web preview-web
   :location (get-new-location (peek strand) slack) ;combine with global-coords
   :orientation (Math/toRadians orientation)
   :slack slack
   :web (into [] (concat web (conj preview-web strand)))
   }
)